from flask import Flask, request, render_template, send_file
import pandas as pd
from openpyxl import Workbook
from openpyxl.styles import Font, PatternFill
from openpyxl.utils import get_column_letter
import os

app = Flask(__name__)

# Rota principal
@app.route('/', methods=['GET', 'POST'])
def upload_file():
    if request.method == 'POST':
        if 'file' not in request.files:
            return 'Nenhum arquivo selecionado'
        file = request.files['file']
        if file.filename == '':
            return 'Nenhum arquivo selecionado'
        if file and file.filename.endswith('.xlsx'):
            # Salva o arquivo enviado
            filename = file.filename
            filepath = os.path.join('uploads', filename)
            file.save(filepath)

            # Processa o arquivo
            processed_filename = process_file(filepath, filename)

            # Envia o arquivo processado para download
            return send_file(os.path.join('processed', processed_filename), as_attachment=True)

        else:
            return 'Formato de arquivo não suportado. Envie um arquivo .xlsx'
    return render_template('upload.html')

# Função de processamento do arquivo
def process_file(filepath, original_filename):
    # Seu código de processamento
    # Modificado para usar o filepath do arquivo enviado
    df = pd.read_excel(filepath, header=None, names=["Data", "Trabalhador"])

    # Agrupa os trabalhadores e as datas trabalhadas por trabalhador
    df_grouped = df.groupby("Trabalhador")["Data"].apply(list).reset_index()

    # Cria uma nova planilha e adiciona o cabeçalho
    wb = Workbook()
    ws = wb.active
    ws.title = "planilhapronta"
    ws.append(["Trabalhador", "Datas Trabalhadas", "Finais de Semana", "Dias Úteis"])

    # Adiciona os dados dos trabalhadores e as datas trabalhadas na planilha
    for index, row in df_grouped.iterrows():
        trabalhador = row["Trabalhador"]
        datas = row["Data"]
        datas_str = ";".join([str(x.day).zfill(2) for x in datas])

        # Verifica se as datas trabalhadas são um sábado ou domingo e preenche em vermelho
        fim_de_semana = 0
        dias_uteis = 0
        fill = None
        for data in datas:
            if data.weekday() == 5 or data.weekday() == 6:
                fim_de_semana += 1
                fill = PatternFill(start_color='FFC7CE', end_color='FFC7CE', fill_type='solid')
            else:
                dias_uteis += 1

        # Adiciona os dados do trabalhador, as datas trabalhadas e a contagem de dias trabalhados em fins de semana e dias úteis
        ws.append([trabalhador, datas_str, fim_de_semana, dias_uteis])
        if fill is not None:
            for cell in ws["B"]:
                if cell.row == ws.max_row:
                    cell.fill = fill

    # Formata a planilha
    ws.column_dimensions[get_column_letter(1)].width = 20
    ws.column_dimensions[get_column_letter(2)].width = 40
    ws.column_dimensions[get_column_letter(3)].width = 20
    ws.column_dimensions[get_column_letter(4)].width = 20
    ws.freeze_panes = "A2"
    ws["A1"].font = Font(bold=True)
    ws["B1"].font = Font(bold=True)
    ws["C1"].font = Font(bold=True)
    ws["D1"].font = Font(bold=True)
    ws.auto_filter.ref = ws.dimensions

    # Salva o arquivo processado com "_pronto" adicionado ao nome
    processed_filename = os.path.splitext(original_filename)[0] + '_pronto.xlsx'
    processed_filepath = os.path.join('processed', processed_filename)
    wb.save(processed_filepath)

    return processed_filename

if __name__ == '__main__':
    # Cria os diretórios se não existirem
    if not os.path.exists('uploads'):
        os.makedirs('uploads')
    if not os.path.exists('processed'):
        os.makedirs('processed')
    app.run(debug=True)
